/*******************************************************************************
* PMC825 UDP/IP Socket Interface Test Program                                  *
*                                                                              *
* (C) 2010 Stock Flight Systems. All rights reserved.                          *
*                                                                              *
* Filename: pmc825canaerospace.c                                               *
*                                                                              *
* Redistribution and use in source and binary forms, with or without           *
* modification, are permitted provided that the following conditions are met:  *
* Redistributions of source code must retain the above copyright notice, this  *
* list of conditions and the following disclaimer.                             *
*                                                                              *
* Redistributions in binary form must reproduce the above copyright notice,    *
* this list of conditions and the following disclaimer in the documentation    *
* and/or other materials provided with the distribution.                       *
*                                                                              *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS" AND ANY EXPRESS    *
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES *
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN   *
* NO EVENT SHALL THE COPYRIGHT OWNER BE LIABLE FOR ANY DIRECT, INDIRECT,       *
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT *
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,  *
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING         *
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, *
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                           *
*                                                                              *
* This file contains a test program which interfaces to the PMC825 via the     *
* UDP/IP socket interface.                                                     *
*                                                                              *
* Function names                                                               *
* ____________________________________________________________________________ *
*                                                                              *
* int main(void)                                                               *
* void ExceptionHandler(void)                                                  *
*                                                                              *
* MODIFICATIONS:                                                               *
*                                                                              *
* When          Version      What                                  Who         *
* ____________________________________________________________________________ *
*                                                                              *
* 26.02.2010    1.0          Initial Version                       M. Stock    *
* 21.04.2010    1.1          Command line interface added          M. Stock    *
* 12.05.2010    1.2          Error frames printed                  M. Stock    *
* 11.06.2010    1.3          Some minor bugs fixed                 M. Stock    *
*                                                                              *
*******************************************************************************/

/*
 * Includes.
 */

#include <math.h>
#include "pmc825socket.h"
#include "can_as.h"

/*
 * Local definitions.
 */

#define	LPORT_BASE	34567           /* Local UDP/IP port number base */
#define	RPORT_BASE      34568           /* PMC825 UDP/IP port number base */

#define PRINT_CAN_MSGS
//#define PRINT_CTRL_MSGS
#define TX_TEST_MSG
//#define RES_TIME

/*
 * Globals.
 */

PMC825_IF Pmc825;

/*
 * ExceptionHandler() frees all resources then terminates.
 */

void ExceptionHandler(void)
{
Pmc825StopInterface(&Pmc825);
printf("\nProgram terminated, all resources released.\n");
exit(0);
}

/*
 * main() starts here.
 */

int main (int argc, char *argv[])
{
#ifndef WIN32
struct timespec t2,t1 = {0,1000000*5};	/* 10ms frame time */
#endif
unsigned int *iptr, ts, tsl, tsh, *test, host_ip, pmc825_ip, lport, rport;
unsigned short tmr_res, *roll_ctrl_cmd, *pitch_ctrl_cmd, *yaw_ctrl_cmd, *coll_ctrl_cmd;
unsigned char *ucptr, tmp[4], hip[32], pip[32], ch[8], init_roll_axis;
int loops, statseq, ret, rxseq, txseq, ctrlseq, timeout, chan, ip[4];
float *fptr, tsf, *roll_torque, *pitch_torque, *yaw_torque, *coll_torque, angle, value;
CAN_AS_MSG rx_buf, ctrl_cmd[4], trim_pos[4], target_tq[4];
CTRL_MSG rx_ctrl, tx_ctrl;
CAN_STAT_REGS *stat;

/*
 * First of all, get IP addresses and CAN channel number.
 */

if (argc != 4)
  {
  printf("usage: %s host_ip pmc825_ip can_channel\n", argv[0]);
  exit(1);
  }

strcpy(hip, argv[1]);
strcpy(pip, argv[2]);
strcpy(ch, argv[3]);

sscanf(hip, "%d.%d.%d.%d", &ip[0], &ip[1], &ip[2], &ip[3]);
host_ip = (ip[0] << 24) | (ip[1] << 16) | (ip[2] << 8) | ip[3];

sscanf(pip, "%d.%d.%d.%d", &ip[0], &ip[1], &ip[2], &ip[3]);
pmc825_ip = (ip[0] << 24) | (ip[1] << 16) | (ip[2] << 8) | ip[3];

sscanf(ch, "%d", &chan);

lport = LPORT_BASE + chan * 2;
rport = RPORT_BASE + chan * 2;

tmr_res = 1 << chan;

/*
 * Set up the exception handler. We want to make sure that we perform
 * cleanup in case we are terminated by some unexpected event.
 */

#ifndef WIN32
signal(SIGQUIT, (void *) ExceptionHandler);
signal(SIGINT,  (void *) ExceptionHandler);
signal(SIGTERM, (void *) ExceptionHandler);
signal(SIGKILL, (void *) ExceptionHandler);
signal(SIGPIPE, (void *) ExceptionHandler);
#endif

/*
 * Initialize the interface for the PMC825.
 */

ret = Pmc825StartInterface(&Pmc825, pmc825_ip, host_ip, rport, lport, chan);

if (ret == PMC825_MEM_ALLOC_ERR)
  {
  printf("Memory allocation error, exiting ...\n");
  Pmc825StopInterface(&Pmc825);
  exit(0);
  }
else if (ret == PMC825_SOCKET_ERR)
  {
  printf("Socket error, exiting ...\n");
  Pmc825StopInterface(&Pmc825);
  exit(0);
  }

/*
 * Set CAN baud rate.
 */
#ifdef XXX
tx_ctrl.opcode = CAN_CTRL;
tx_ctrl.svc_rsp_code = INIT_CAN_CHIP;
tx_ctrl.arg[0] = CAN_1M;
tx_ctrl.arg[1] = 0;
tx_ctrl.arg[2] = 0;
tx_ctrl.arg[3] = 0;
tx_ctrl.arg[4] = 0;

ret = Pmc825CtrlWrite(&Pmc825, &tx_ctrl);

if (ret != PMC825_OK)
  {
  printf("INIT_CAN_CHIP failed, exiting ...\n");
  Pmc825StopInterface(&Pmc825);
  exit(0);
  }
#endif
/*
 * Setup the CANaerospace transmit messages.
 */

ctrl_cmd[0].identifier = 1505; /* Roll axis */
ctrl_cmd[0].byte_count = 6;
ctrl_cmd[0].frame_type = DATA;
ctrl_cmd[0].node_id = 51; /* FIXME */
ctrl_cmd[0].data_type = AS_BSHORT;
ctrl_cmd[0].service_code = 0;
ctrl_cmd[0].msg_code = 0;
ctrl_cmd[0].can_status = 0;
ctrl_cmd[0].error_counter = 0;
ctrl_cmd[0].time_stamp_lo = 0;
ctrl_cmd[0].time_stamp_hi = 0;

roll_ctrl_cmd = (unsigned short *) &(ctrl_cmd[0].data[0]);
*roll_ctrl_cmd = 0;

ctrl_cmd[1].identifier = 1555; /* Pitch axis */
ctrl_cmd[1].byte_count = 6;
ctrl_cmd[1].frame_type = DATA;
ctrl_cmd[1].node_id = 52; /* FIXME */
ctrl_cmd[1].data_type = AS_BSHORT;
ctrl_cmd[1].service_code = 0;
ctrl_cmd[1].msg_code = 0;
ctrl_cmd[1].can_status = 0;
ctrl_cmd[1].error_counter = 0;
ctrl_cmd[1].time_stamp_lo = 0;
ctrl_cmd[1].time_stamp_hi = 0;

pitch_ctrl_cmd = (unsigned short *) &(ctrl_cmd[1].data[0]);
*pitch_ctrl_cmd = 0;

ctrl_cmd[2].identifier = 1605; /* Yaw axis */
ctrl_cmd[2].byte_count = 6;
ctrl_cmd[2].frame_type = DATA;
ctrl_cmd[2].node_id = 53; /* FIXME */
ctrl_cmd[2].data_type = AS_BSHORT;
ctrl_cmd[2].service_code = 0;
ctrl_cmd[2].msg_code = 0;
ctrl_cmd[2].can_status = 0;
ctrl_cmd[2].error_counter = 0;
ctrl_cmd[2].time_stamp_lo = 0;
ctrl_cmd[2].time_stamp_hi = 0;

yaw_ctrl_cmd = (unsigned short *) &(ctrl_cmd[2].data[0]);
*yaw_ctrl_cmd = 0;

ctrl_cmd[3].identifier = 1655; /* Collective axis */
ctrl_cmd[3].byte_count = 6;
ctrl_cmd[3].frame_type = DATA;
ctrl_cmd[3].node_id = 54; /* FIXME */
ctrl_cmd[3].data_type = AS_BSHORT;
ctrl_cmd[3].service_code = 0;
ctrl_cmd[3].msg_code = 0;
ctrl_cmd[3].can_status = 0;
ctrl_cmd[3].error_counter = 0;
ctrl_cmd[3].time_stamp_lo = 0;
ctrl_cmd[3].time_stamp_hi = 0;

coll_ctrl_cmd = (unsigned short *) &(ctrl_cmd[3].data[0]);
*coll_ctrl_cmd = 0;

target_tq[0].identifier = 1510; /* Roll axis */
target_tq[0].byte_count = 8;
target_tq[0].frame_type = DATA;
target_tq[0].node_id = 51; /* FIXME */
target_tq[0].data_type = AS_FLOAT;
target_tq[0].service_code = 0;
target_tq[0].msg_code = 0;
target_tq[0].can_status = 0;
target_tq[0].error_counter = 0;
target_tq[0].time_stamp_lo = 0;
target_tq[0].time_stamp_hi = 0;

roll_torque = (float *) &(target_tq[0].data[0]);
*roll_torque = 0.0;

target_tq[1].identifier = 1560; /* Pitch axis */
target_tq[1].byte_count = 8;
target_tq[1].frame_type = DATA;
target_tq[1].node_id = 52; /* FIXME */
target_tq[1].data_type = AS_FLOAT;
target_tq[1].service_code = 0;
target_tq[1].msg_code = 0;
target_tq[1].can_status = 0;
target_tq[1].error_counter = 0;
target_tq[1].time_stamp_lo = 0;
target_tq[1].time_stamp_hi = 0;

pitch_torque = (float *) &(target_tq[1].data[0]);
*pitch_torque = 0.0;

target_tq[2].identifier = 1610; /* Yaw axis */
target_tq[2].byte_count = 8;
target_tq[2].frame_type = DATA;
target_tq[2].node_id = 53; /* FIXME */
target_tq[2].data_type = AS_FLOAT;
target_tq[2].service_code = 0;
target_tq[2].msg_code = 0;
target_tq[2].can_status = 0;
target_tq[2].error_counter = 0;
target_tq[2].time_stamp_lo = 0;
target_tq[2].time_stamp_hi = 0;

yaw_torque = (float *) &(target_tq[2].data[0]);
*yaw_torque = 0.0;

target_tq[3].identifier = 1660; /* Collective axis */
target_tq[3].byte_count = 8;
target_tq[3].frame_type = DATA;
target_tq[3].node_id = 54; /* FIXME */
target_tq[3].data_type = AS_FLOAT;
target_tq[3].service_code = 0;
target_tq[3].msg_code = 0;
target_tq[3].can_status = 0;
target_tq[3].error_counter = 0;
target_tq[3].time_stamp_lo = 0;
target_tq[3].time_stamp_hi = 0;

coll_torque = (float *) &(target_tq[3].data[0]);
*coll_torque = 0.0;

/*
 * Initialize other variables.
 */

rxseq = 0;
txseq = 0;
ctrlseq = 0;
statseq = 0;
timeout = 0;

angle = 0.0;

init_roll_axis = 0;

/*
 * Initialize the CLS axes.
 */

for (loops = 0; loops < 4; loops++)
  ctrl_cmd[loops].msg_code++;

*roll_ctrl_cmd = 0x0220;
*pitch_ctrl_cmd = 0x0220;
*yaw_ctrl_cmd = 0x0220;
*coll_ctrl_cmd = 0x0220;

Pmc825CanAerospaceWrite(&Pmc825, &ctrl_cmd[0], 4);

/*
 * Wait for positive response of all axes.
 */

do
  {
   ret = Pmc825CanAerospaceRead(&Pmc825, &rx_buf);

  while (ret != PMC825_NO_MSG)
    {
    if (rx_buf.identifier == 1500) /* Roll axis actual position */
      {
      if ((rx_buf.service_code & 0x0f) == 0x0f) /* Homing completed */
        init_roll_axis = rx_buf.service_code;
      }
    ret = Pmc825CanAerospaceRead(&Pmc825, &rx_buf);
    }

  nanosleep(&t1,&t2);
  timeout++;
  } while ((timeout < 10000) && (init_roll_axis != 0x0f));

if (timeout >= 10000)
  printf("Init failed [0x%02x]\n", init_roll_axis);

/*
 * Now go into an endless loop reading from and writing to the PMC825.
 */

printf("Start processing PMC825 data CAN %d [Host IP %s]\n", chan, hip);

for (;;)
  {
  /*
   * Compute sine wave.
   */

//   sine = (float *) &(tx_buf[loops].data[0]);
//   *sine = (float) sin((double) angle);
//   angle += 0.062832;

  value = sinf(angle);
  angle += 0.01;
 
  /*
   * Get received CAN control messages and print them.
   */

  ret = Pmc825CtrlRead(&Pmc825, &rx_ctrl);

  while (ret != PMC825_NO_MSG)
    {
#ifdef PRINT_CTRL_MSGS
    switch (rx_ctrl.opcode)
      {
      case CAN_WRITE_RSP:
	printf("Ethernet WRITE_RSP Packet\n");
	printf("-------------------------\n");
	printf("Response Code     = %d\n", rx_ctrl.svc_rsp_code);
	printf("Messages written  = %d\n", rx_ctrl.arg[0]);
        break;

      case CAN_CTRL_RSP:
	printf("Ethernet CTRL_RSP Packet\n");
	printf("------------------------\n");
	printf("Response Code  = %d\n", rx_ctrl.svc_rsp_code);

        for (loops = 0; loops < 64; loops++)
          printf("arg[%03d]  = 0x%04x\n", loops, rx_ctrl.arg[loops]);

        printf("\n");
        break;

      case CAN_STATUS:
        if (statseq < 50)
          statseq++;
        else
          {
          statseq = 0;
          stat = (CAN_STAT_REGS *) &(rx_ctrl.arg[0]);

          printf("\nEthernet STATUS Packet\n");
          printf("----------------------\n");
          printf("PMC825 Name       = %s\n",stat->name);
          printf("PMC825 IP Address = %d.%d.%d.%d\n",stat->ip_addr[0],
                                                     stat->ip_addr[1],
                                                     stat->ip_addr[2],
                                                     stat->ip_addr[3]);
          printf("CAN Channel       = %d\n",chan);
          printf("Bus Timing Reg    = 0x%04x\n",stat->btr);
          printf("CAN Status        = 0x%04x\n",stat->can_status);
          printf("Error Counters    = 0x%04x\n",stat->error_counter);
          printf("CPM Setting       = 0x%04x\n",stat->cpm_mode);
          printf("TX Msgs (100ms)   = %d\n",    stat->tx_msgs);
          printf("RX Msgs (100ms)   = %d\n",    stat->rx_msgs);
          printf("TX Bits (100ms)   = %d\n",    stat->tx_bits);
          printf("RX Bits (100ms)   = %d\n",    stat->rx_bits);
          printf("Board Temperature = %d\n",    stat->t1);
          printf("FPGA Temperature  = %d\n",    stat->t2);
          sleep(1);
          }
        break;

      default:
	printf("Unknown Ethernet Packet\n");
	printf("-----------------------\n");
	printf("Unknown Opcode    = %d\n", rx_ctrl.opcode);
	printf("Service Code      = 0x%04x\n", rx_ctrl.svc_rsp_code);
        break;
      }
#endif
    ret = Pmc825CtrlRead(&Pmc825, &rx_ctrl);
    }

  /*
   * Get all received CANaerospace messages and print them.
   */

  ret = Pmc825CanAerospaceRead(&Pmc825, &rx_buf);

  while (ret != PMC825_NO_MSG)
    {
#ifdef PRINT_CAN_MSGS
    /*
     * Artificially reduce the time stamp resolution from 31,25ns to 1us
     * for display purposes.
     */

    tsl = rx_buf.time_stamp_lo;
    tsh = rx_buf.time_stamp_hi;
    tsl = (tsl >> 5) & 0x07ffffff;
    tsh = (tsh << 27) & 0xf8000000;
    ts = tsl | tsh;
    tsf = ((float) ts)/1e6;

    if (rx_buf.frame_type == (char) ERROR_FRAME)
      {
      printf("%1.4f  Error Frame\n", tsf);
      }
    else
      {
      if ((rx_buf.identifier & EXT_ID) == EXT_ID)
	{
	rx_buf.identifier &= ~EXT_ID;
	printf("%1.4f  EXT-ID $%08X [%03d %03d %03d %03d] -> ",
               tsf, rx_buf.identifier, rx_buf.node_id, rx_buf.data_type,
	       (rx_buf.service_code & 0xff), (rx_buf.msg_code & 0xff));
	}
      else
	printf("%1.4f  STD-ID %04d [%03d %03d %03d %03d] -> ",
               tsf, rx_buf.identifier, rx_buf.node_id, rx_buf.data_type,
	       (rx_buf.service_code & 0xff), (rx_buf.msg_code & 0xff));

      if (rx_buf.data_type == AS_FLOAT)
	{
#ifdef LITTLE_ENDIAN
	tmp[0] = rx_buf.data[0];
	tmp[1] = rx_buf.data[1];
	tmp[2] = rx_buf.data[2];
	tmp[3] = rx_buf.data[3];

	rx_buf.data[0] = tmp[3];
	rx_buf.data[1] = tmp[2];
	rx_buf.data[2] = tmp[1];
	rx_buf.data[3] = tmp[0];
#endif
	fptr = (float *) &(rx_buf.data[0]);
	printf("Data = %.1f\n", *fptr);
	}
      else if ((rx_buf.data_type == AS_LONG) ||
               (rx_buf.data_type == AS_ULONG) ||
               (rx_buf.data_type == AS_BLONG))
	{
#ifdef LITTLE_ENDIAN
	tmp[0] = rx_buf.data[0];
	tmp[1] = rx_buf.data[1];
	tmp[2] = rx_buf.data[2];
	tmp[3] = rx_buf.data[3];

	rx_buf.data[0] = tmp[3];
	rx_buf.data[1] = tmp[2];
	rx_buf.data[2] = tmp[1];
	rx_buf.data[3] = tmp[0];
#endif
	iptr = (unsigned int *) &(rx_buf.data[0]);
	printf("Data = %08x\n", *iptr);
	}
      else
	{
	for (loops = 4; loops < rx_buf.byte_count; loops++)
	  printf("0x%02X ", rx_buf.data[loops] & 0xff);
	printf("\n");
	}
      }
#endif
    ret = Pmc825CanAerospaceRead(&Pmc825, &rx_buf);
    }

  /*
   * Transmit a CANaerospace message every second.
   */

#ifdef TX_TEST_MSG
  if (txseq < 10)
    txseq++;
  else
    {
    txseq = 0;
    *roll_torque = value;

#ifdef LITTLE_ENDIAN
    tmp[0] = target_tq[0].data[0];
    tmp[1] = target_tq[0].data[1];
    tmp[2] = target_tq[0].data[2];
    tmp[3] = target_tq[0].data[3];

    target_tq[0].data[0] = tmp[3];
    target_tq[0].data[1] = tmp[2];
    target_tq[0].data[2] = tmp[1];
    target_tq[0].data[3] = tmp[0];
#endif

    target_tq[0].msg_code++;

    Pmc825CanAerospaceWrite(&Pmc825, &target_tq[0], 1);
    }
#endif

  /*
   * Reset the time stamp every 10 seconds.
   */

#ifdef RES_TIME
  if (ctrlseq < 1000)
    ctrlseq++;
  else
    {
    ctrlseq = 0;
    tx_ctrl.opcode = CAN_CTRL;
    tx_ctrl.svc_rsp_code = RESET_TIME_STAMP;
    tx_ctrl.arg[0] = tmr_res;

    Pmc825CtrlWrite(&Pmc825, &tx_ctrl);
    }
#endif

  /*
   * 10ms time frame.
   */

#ifndef WIN32
  nanosleep(&t1,&t2);
#else
  Sleep (10);
#endif
  }
exit(0);
}

/*
 * End of file.
 */
